// --- Lógica de Sesión ---
const creditCounter = document.getElementById('creditCounterDisplay');
const userSessionElement = document.getElementById('user-session');

function updateCreditCounter(amount) {
    if (creditCounter) { creditCounter.textContent = `Créditos: ${amount}`; creditCounter.style.display = 'block'; }
}

function checkLoginStatus() {
    fetch('check_session.php', { cache: 'no-store' })
        .then(r => r.json()).then(d => {
            if (d.loggedIn) {
                userSessionElement.innerHTML = `Hola, <strong>${d.nombre}</strong> | <a href="index.html">Inicio</a> | <a href="logout.php">Salir</a>`;
                updateCreditCounter(d.creditos);
            } else { alert('Inicia sesión'); window.location.href = 'index.html'; }
        }).catch(e => console.error(e));
}
checkLoginStatus();

document.addEventListener('DOMContentLoaded', () => {
    // Elementos DOM
    const charInput = document.getElementById('charInput');
    const font3dSelector = document.getElementById('font3dSelector');
    const letterHeightInput = document.getElementById('letterHeight');
    const letterDepthInput = document.getElementById('letterDepth');
    const tabSizeInput = document.getElementById('tabSizeInput'); 
    const paperSizeSelector = document.getElementById('paperSizeSelector');
    const generate3dPdfBtn = document.getElementById('generate3dPdfBtn');
    const statusDiv = document.getElementById('status');
    
    // Elementos de Preview
    const previewCanvas = document.getElementById('previewCanvas');
    const ctx = previewCanvas.getContext('2d');
    const dimensionsDisplay = document.getElementById('dimensionsDisplay');

    const fontCache = new Map();
    let currentFont = null;

    // --- URLs SEGURAS (JSDelivr) ---
    const FONT_URLS = {
        'Anton': 'https://cdn.jsdelivr.net/gh/google/fonts@main/ofl/anton/Anton-Regular.ttf',
        'Bebas Neue': 'https://cdn.jsdelivr.net/gh/google/fonts@main/ofl/bebasneue/BebasNeue-Regular.ttf',
        'Oswald': 'https://cdn.jsdelivr.net/gh/google/fonts@main/ofl/oswald/static/Oswald-Bold.ttf',
        'Alfa Slab One': 'https://cdn.jsdelivr.net/gh/google/fonts@main/ofl/alfaslabone/AlfaSlabOne-Regular.ttf',
    };

    // --- CARGA DE FUENTES ---
    async function loadFont(fontName) {
        if (fontCache.has(fontName)) return fontCache.get(fontName);
        
        statusDiv.textContent = `Descargando fuente ${fontName}...`;
        statusDiv.style.color = "#007bff"; 

        try {
            const response = await fetch(FONT_URLS[fontName]);
            if (!response.ok) throw new Error(`Error HTTP ${response.status}`);
            const buffer = await response.arrayBuffer();
            const font = opentype.parse(buffer);
            fontCache.set(fontName, font);
            statusDiv.textContent = 'Fuente cargada. Listo.';
            statusDiv.style.color = "#28a745"; 
            return font;
        } catch (e) {
            console.error(e);
            statusDiv.textContent = "Error cargando fuente. Revisa tu conexión.";
            statusDiv.style.color = "#dc3545"; 
            return null;
        }
    }

    // --- FUNCIÓN DE PREVISUALIZACIÓN (CORREGIDA: CENTRADO ABSOLUTO) ---
    function updatePreview() {
        if (!currentFont) return;

        const char = charInput.value.toUpperCase() || 'A';
        const heightCM = parseFloat(letterHeightInput.value) || 0;
        
        // 1. Limpiar Canvas
        ctx.clearRect(0, 0, previewCanvas.width, previewCanvas.height);
        
        // 2. Obtener medidas base de la fuente (sin escalar)
        // Usamos un tamaño arbitrario grande (100) para medir proporciones
        const measureSize = 100;
        const tempPath = currentFont.getPath(char, 0, 0, measureSize);
        const box = tempPath.getBoundingBox();
        
        const fontW = box.x2 - box.x1;
        const fontH = box.y2 - box.y1;
        
        // 3. Calcular Medidas Reales (CM) para mostrar al usuario
        const aspectRatio = fontW / fontH;
        const calculatedWidthCM = heightCM * aspectRatio;

        if (dimensionsDisplay) {
            dimensionsDisplay.innerHTML = `
                Alto: ${heightCM} cm <br>
                Ancho aprox: <span style="font-size:1.2em; color:#d32f2f;">${calculatedWidthCM.toFixed(1)} cm</span>
            `;
        }

        // 4. CALCULAR ESCALA PARA EL CANVAS
        // Dejamos un margen de 40px alrededor
        const margin = 40;
        const availableW = previewCanvas.width - (margin * 2);
        const availableH = previewCanvas.height - (margin * 2);

        // Factor de escala: ¿Cuánto debo agrandar/achicar la letra base (size 100) para que quepa?
        const scaleX = availableW / fontW;
        const scaleY = availableH / fontH;
        const scaleFactor = Math.min(scaleX, scaleY); // Usamos el menor para que quepa todo

        const finalFontSize = measureSize * scaleFactor; // Este es el tamaño de fuente en px para el canvas

        // 5. CALCULAR POSICIÓN DE DIBUJO (CENTRADO PERFECTO)
        // Obtenemos el path final con el tamaño correcto, pero en posición 0,0
        const finalPathTemp = currentFont.getPath(char, 0, 0, finalFontSize);
        const finalBox = finalPathTemp.getBoundingBox();
        
        const finalW = finalBox.x2 - finalBox.x1;
        const finalH = finalBox.y2 - finalBox.y1;

        // Calculamos el centro del bounding box de la letra
        const textCenterX = (finalBox.x1 + finalBox.x2) / 2;
        const textCenterY = (finalBox.y1 + finalBox.y2) / 2;

        // Calculamos el centro del canvas
        const canvasCenterX = previewCanvas.width / 2;
        const canvasCenterY = previewCanvas.height / 2;

        // La posición de dibujo es: CentroCanvas - CentroLetra
        const drawX = canvasCenterX - textCenterX;
        const drawY = canvasCenterY - textCenterY;

        // 6. DIBUJAR
        ctx.fillStyle = "#e3f2fd";
        ctx.strokeStyle = "#007bff";
        ctx.lineWidth = 2;

        const pathVisual = currentFont.getPath(char, drawX, drawY, finalFontSize);
        drawOpenTypePathOnCanvas(ctx, pathVisual);
        ctx.fill();
        ctx.stroke();

        // 7. DIBUJAR COTAS (FLECHAS DE MEDIDA)
        // Las dibujamos relativas a la caja de la letra
        const boxLeft = drawX + finalBox.x1;
        const boxRight = drawX + finalBox.x2;
        const boxTop = drawY + finalBox.y1;
        const boxBottom = drawY + finalBox.y2;

        // Vertical (Izquierda)
        drawDimensionLine(ctx, boxLeft - 20, boxBottom, boxLeft - 20, boxTop, `${heightCM}cm`);
        // Horizontal (Abajo)
        drawDimensionLine(ctx, boxLeft, boxBottom + 20, boxRight, boxBottom + 20, `${calculatedWidthCM.toFixed(1)}cm`);
    }

    function drawOpenTypePathOnCanvas(ctx, path) {
        ctx.beginPath();
        for (let cmd of path.commands) {
            if (cmd.type === 'M') ctx.moveTo(cmd.x, cmd.y);
            else if (cmd.type === 'L') ctx.lineTo(cmd.x, cmd.y);
            else if (cmd.type === 'Q') ctx.quadraticCurveTo(cmd.x1, cmd.y1, cmd.x, cmd.y);
            else if (cmd.type === 'C') ctx.bezierCurveTo(cmd.x1, cmd.y1, cmd.x2, cmd.y2, cmd.x, cmd.y);
            else if (cmd.type === 'Z') ctx.closePath();
        }
    }

    function drawDimensionLine(ctx, x1, y1, x2, y2, text) {
        ctx.save();
        ctx.strokeStyle = "#666";
        ctx.fillStyle = "#666";
        ctx.font = "bold 14px Arial";
        ctx.lineWidth = 1;
        ctx.beginPath();
        ctx.moveTo(x1, y1); ctx.lineTo(x2, y2);
        
        if(Math.abs(x1-x2) < 1) { // Vertical
            ctx.moveTo(x1-3, y1+3); ctx.lineTo(x1, y1); ctx.lineTo(x1+3, y1+3); // Flecha abajo
            ctx.moveTo(x2-3, y2-3); ctx.lineTo(x2, y2); ctx.lineTo(x2+3, y2-3); // Flecha arriba
            ctx.fillText(text, x1-50, (y1+y2)/2);
        } else { // Horizontal
            ctx.moveTo(x1+3, y1-3); ctx.lineTo(x1, y1); ctx.lineTo(x1+3, y1+3); // Flecha izq
            ctx.moveTo(x2-3, y2-3); ctx.lineTo(x2, y2); ctx.lineTo(x2-3, y2+3); // Flecha der
            ctx.fillText(text, (x1+x2)/2 - 15, y1+20);
        }
        ctx.stroke();
        ctx.restore();
    }

    // --- EVENT LISTENERS UI ---
    const inputs = [charInput, letterHeightInput, letterDepthInput, tabSizeInput];
    inputs.forEach(el => el.addEventListener('input', updatePreview));
    
    font3dSelector.addEventListener('change', async () => {
        currentFont = await loadFont(font3dSelector.value);
        updatePreview();
    });

    // --- INICIALIZACIÓN ---
    (async function init() {
        currentFont = await loadFont('Anton');
        updatePreview();
    })();


    // --- MATEMÁTICA DE CURVAS ---
    function getCubicBezierLength(p0x, p0y, p1x, p1y, p2x, p2y, p3x, p3y) {
        let length = 0, px = p0x, py = p0y;
        const segments = 20;
        for (let i = 1; i <= segments; i++) {
            let t = i / segments, u = 1 - t;
            let x = Math.pow(u,3)*p0x + 3*Math.pow(u,2)*t*p1x + 3*u*Math.pow(t,2)*p2x + Math.pow(t,3)*p3x;
            let y = Math.pow(u,3)*p0y + 3*Math.pow(u,2)*t*p1y + 3*u*Math.pow(t,2)*p2y + Math.pow(t,3)*p3y;
            length += Math.sqrt(Math.pow(x - px, 2) + Math.pow(y - py, 2));
            px = x; py = y;
        }
        return length;
    }
    function getQuadBezierLength(p0x, p0y, p1x, p1y, p2x, p2y) {
        let length = 0, px = p0x, py = p0y;
        const segments = 20;
        for (let i = 1; i <= segments; i++) {
            let t = i / segments, u = 1 - t;
            let x = Math.pow(u,2)*p0x + 2*u*t*p1x + Math.pow(t,2)*p2x;
            let y = Math.pow(u,2)*p0y + 2*u*t*p1y + Math.pow(t,2)*p2y;
            length += Math.sqrt(Math.pow(x - px, 2) + Math.pow(y - py, 2));
            px = x; py = y;
        }
        return length;
    }
    function distance(x1, y1, x2, y2) { return Math.sqrt(Math.pow(x2-x1,2) + Math.pow(y2-y1,2)); }

    // --- GENERACIÓN PDF ---
    generate3dPdfBtn.addEventListener('click', async () => {
        if(!currentFont) {
            currentFont = await loadFont(font3dSelector.value);
            if(!currentFont) { alert("No se pudo cargar la fuente. Verifica tu internet."); return; }
        }
        statusDiv.textContent = "Calculando geometría...";
        
        const { jsPDF } = window.jspdf;
        const doc = new jsPDF({unit: 'mm', format: paperSizeSelector.value});
        
        const hCM = parseFloat(letterHeightInput.value);
        const dCM = parseFloat(letterDepthInput.value);
        const tabSizeCM = parseFloat(tabSizeInput.value);
        
        const hMM = hCM * 10;
        const dMM = dCM * 10;
        const tabSizeMM = tabSizeCM * 10;
        
        const char = charInput.value.toUpperCase();
        const path = currentFont.getPath(char, 0, 0, 72);
        const box = path.getBoundingBox();
        const scale = hMM / (box.y2 - box.y1);
        const wMM = (box.x2 - box.x1) * scale;

        doc.setFontSize(20); doc.text(`Molde Letra 3D: ${char}`, 20, 20);
        doc.setFontSize(12); doc.text(`Alto: ${hCM}cm | Profundidad: ${dCM}cm`, 20, 30);
        doc.text(`Fuente: ${font3dSelector.value}`, 20, 40);
        doc.text("tiktok: @rafooo90", 20, 280);

        const drawTiledPath = (path, scale, totalW, totalH, margin, isFlipped) => {
            const pageW = doc.internal.pageSize.getWidth();
            const pageH = doc.internal.pageSize.getHeight();
            const contentW = pageW - (margin * 2);
            const contentH = pageH - (margin * 2);
            const cols = Math.ceil(totalW / contentW);
            const rows = Math.ceil(totalH / contentH);

            for (let r = 0; r < rows; r++) {
                for (let c = 0; c < cols; c++) {
                    doc.addPage();
                    doc.setFontSize(10);
                    doc.text(`Fila ${r+1}-${c+1} (${isFlipped ? 'TRASERA' : 'FRONTAL'})`, margin, margin-2);
                    doc.text(`${char}`, pageW-20, margin-2);
                    
                    doc.setDrawColor(180); doc.setLineDash([3,3]);
                    doc.rect(margin, margin, contentW, contentH);
                    doc.setLineDash([]); doc.setDrawColor(0); doc.setLineWidth(0.4);

                    doc.saveGraphicsState();
                    doc.rect(margin, margin, contentW, contentH);
                    doc.clip();

                    const letterX = margin - (c * contentW);
                    const letterY = margin - (r * contentH) + totalH; 
                    const scaleX = isFlipped ? -scale : scale;
                    
                    for (let cmd of path.commands) {
                        const tx = (v) => letterX + (v * scaleX) + (isFlipped ? totalW : 0);
                        const ty = (v) => letterY - (v * scale); 
                        if (cmd.type === 'M') doc.moveTo(tx(cmd.x), ty(cmd.y));
                        else if (cmd.type === 'L') doc.lineTo(tx(cmd.x), ty(cmd.y));
                        else if (cmd.type === 'C') doc.curveTo(tx(cmd.x1), ty(cmd.y1), tx(cmd.x2), ty(cmd.y2), tx(cmd.x), ty(cmd.y));
                        else if (cmd.type === 'Q') doc.curveTo(tx(cmd.x1), ty(cmd.y1), tx(cmd.x1), ty(cmd.y1), tx(cmd.x), ty(cmd.y));
                        else if (cmd.type === 'Z') doc.close();
                    }
                    doc.stroke();
                    doc.restoreGraphicsState();
                }
            }
        };

        drawTiledPath(path, scale, wMM, hMM, 10, false);
        drawTiledPath(path, scale, wMM, hMM, 10, true);

        const addSideStrips = () => {
            const pageW = doc.internal.pageSize.getWidth();
            const pageH = doc.internal.pageSize.getHeight();
            const margin = 10;
            let curX = margin, curY = margin + 15;
            let piece = 1;

            doc.addPage(); doc.text('Tiras Laterales (Profundidad)', margin, margin);

            let px=0, py=0, sx=0, sy=0;
            for (let cmd of path.commands) {
                let len = 0;
                if (cmd.type === 'M') { px=cmd.x; py=cmd.y; sx=cmd.x; sy=cmd.y; continue; }
                if (cmd.type === 'L') len = distance(px, py, cmd.x, cmd.y);
                if (cmd.type === 'C') len = getCubicBezierLength(px, py, cmd.x1, cmd.y1, cmd.x2, cmd.y2, cmd.x, cmd.y);
                if (cmd.type === 'Q') len = getQuadBezierLength(px, py, cmd.x1, cmd.y1, cmd.x, cmd.y);
                if (cmd.type === 'Z') len = distance(px, py, sx, sy);
                if (cmd.x !== undefined) { px=cmd.x; py=cmd.y; }
                if (len === 0) continue;

                let realLen = len * scale;
                while (realLen > 0) {
                    let drawLen = Math.min(realLen, pageW - (margin*2));
                    if (curY + dMM + (tabSizeMM*2) > pageH - margin) { 
                        doc.addPage(); curY = margin + 15; curX = margin; 
                    }
                    
                    doc.setLineWidth(0.3); doc.setDrawColor(0);
                    doc.rect(curX, curY, drawLen, dMM);
                    doc.setFontSize(8); doc.text(`P${piece}`, curX+2, curY+dMM/2);

                    const tabW = tabSizeMM * 1.5; 
                    const tabH = tabSizeMM;       
                    
                    doc.setLineWidth(0.1);
                    for(let tx = curX; tx < curX + drawLen; tx += tabW) {
                        let tWidth = Math.min(tabW, (curX + drawLen) - tx);
                        if(tWidth > 2) {
                            doc.line(tx, curY, tx + (tWidth/2), curY - tabH); 
                            doc.line(tx + (tWidth/2), curY - tabH, tx + tWidth, curY);
                        }
                    }
                    for(let tx = curX; tx < curX + drawLen; tx += tabW) {
                        let tWidth = Math.min(tabW, (curX + drawLen) - tx);
                        if(tWidth > 2) {
                            doc.line(tx, curY + dMM, tx + (tWidth/2), curY + dMM + tabH); 
                            doc.line(tx + (tWidth/2), curY + dMM + tabH, tx + tWidth, curY + dMM); 
                        }
                    }
                    realLen -= drawLen;
                    curY += dMM + (tabSizeMM*2) + 10; 
                    piece++;
                }
            }
        };

        addSideStrips();
        doc.save(`Letra3D_${char}_${hCM}cm.pdf`);
        statusDiv.textContent = "¡PDF Descargado!";
    });
});