// Asegurarnos de que jsPDF esté disponible globalmente
const { jsPDF } = window.jspdf;
// Cropper no se adjunta a window, pero se importa globalmente desde el script

// --- Constantes de Tamaño de Papel (en milímetros) ---
const PAPER_SIZES_MM = {
    a4: { w: 210, h: 297 },
    letter: { w: 215.9, h: 279.4 }
};
const RASTER_DOT_SIZE = 8; // Tamaño de los puntos para el efecto Raster (en píxeles)

// --- ¡NUEVO! Margen de impresión que ASUMIMOS ---
// 10mm (1cm) en cada lado. Aquí es donde dibujaremos las líneas de corte.
const PRINT_MARGIN_MM = 10; 


// --- DOM Elements ---
const pageSubtitle = document.getElementById('page-subtitle');
const imageLoader = document.getElementById('imageLoader');
const imagePreview = document.getElementById('imagePreview');
const previewContainer = document.getElementById('preview-container');
const previewPlaceholder = document.getElementById('previewPlaceholder');
const gridPreviewOverlay = document.getElementById('gridPreviewOverlay'); 

const filterSelect = document.getElementById('filterSelect'); 
const posterWidthInput = document.getElementById('posterWidth');
const posterHeightInput = document.getElementById('posterHeight');
const aspectLock = document.getElementById('aspectLock');

const paperSizeSelect = document.getElementById('paperSize');
const orientationSelect = document.getElementById('orientation');

// const overlapSelect = document.getElementById('overlapSize'); // Eliminado
const addCutLinesCheckbox = document.getElementById('addCutLines');
const addPageNumbersCheckbox = document.getElementById('addPageNumbers');
const addSummaryPageCheckbox = document.getElementById('addSummaryPage');
const addWatermarkCheckbox = document.getElementById('addWatermark');

const generateBtn = document.getElementById('generateBtn');
const statusDiv = document.getElementById('status');
const creditCounter = document.getElementById('creditCounterDisplay'); 
const userSessionElement = document.getElementById('user-session'); 

// --- Elementos de Recorte (Crop) ---
const cropBtn = document.getElementById('cropBtn');
const cropModal = document.getElementById('cropModal');
const imageToCrop = document.getElementById('imageToCrop');
const saveCropBtn = document.getElementById('saveCropBtn');
const cancelCropBtn = document.getElementById('cancelCropBtn');
let cropper = null;

// --- Elementos de Modal de Créditos ---
const creditsModal = document.getElementById('creditsModal');
const closeCreditsModalBtn = document.getElementById('closeCreditsModalBtn');

// --- Elementos de Guardar Proyecto ---
const saveProjectContainer = document.getElementById('save-project-box');
const projectNameInput = document.getElementById('projectName');
const makePublicCheckbox = document.getElementById('makePublicCheckbox');
const saveProjectBtn = document.getElementById('saveProjectBtn');

// --- Variables Globales ---
let uploadedImage = null; // Objeto Imagen
let originalImageDataUrl = null; // DataURL de la imagen (original o recortada)
let originalImageFile = null; // El objeto Archivo (File) para poder guardarlo
let imageAspectRatio = null;
let calculatedPagesWide = 0;
let calculatedPagesHigh = 0;

// --- Funcionalidad adicional: Desactivar clic derecho ---
document.addEventListener('contextmenu', (e) => {
    e.preventDefault();
    alert('Clic derecho desactivado para proteger el contenido de la página.');
});

// --- LÓGICA DE SESIÓN Y CRÉDITOS ---

function updateCreditCounter(amount) {
    if (creditCounter) {
        creditCounter.textContent = `Créditos: ${amount}`;
        creditCounter.style.display = 'block';
    }
}

function checkLoginStatus() {
    fetch('check_session.php', { cache: 'no-store' }) 
        .then(response => {
            if (!response.ok) {
                throw new Error(`Error HTTP! estado: ${response.status}`);
            }
            return response.json();
        })
        .then(data => {
            if (data.loggedIn) {
                userSessionElement.innerHTML = `
                    <div class="user-info">
                        Bienvenido, <strong>${data.nombre}</strong>
                        <span>|</span>
                        <a href="index.html">Inicio</a>
                        <span>|</span>
                        <a href="galeria.html">Mi Galería</a>
                        <span>|</span>
                        <a href="comunidad.html">Galería Pública</a> 
                        <span>|</span>
                        <a href="logout.php">Cerrar Sesión</a>
                    </div>
                `;
                updateCreditCounter(data.creditos);
            } else {
                alert('Necesitas iniciar sesión para usar esta herramienta.');
                window.location.href = 'index.html';
            }
        })
        .catch(error => {
            console.error('Error al chequear la sesión:', error);
            alert('Error de conexión o de sesión. Redirigiendo al inicio.');
            window.location.href = 'index.html';
        });
}

// --- LÓGICA DE CARGAR PROYECTO ---
function checkAndLoadProject() {
    const urlParams = new URLSearchParams(window.location.search);
    const projectIdToLoad = urlParams.get('cargar_id');

    if (projectIdToLoad) {
        pageSubtitle.textContent = 'Cargando tu proyecto guardado...';
        statusDiv.textContent = 'Cargando proyecto...';

        imageLoader.disabled = true;
        
        fetch(`cargar_proyecto.php?id=${projectIdToLoad}`, { cache: 'no-store' })
            .then(response => response.json())
            .then(data => {
                if (data.success && data.proyecto) {
                    const proj = data.proyecto;
                    
                    posterWidthInput.value = parseFloat(proj.ancho_cm).toFixed(1);
                    posterHeightInput.value = parseFloat(proj.alto_cm).toFixed(1);
                    filterSelect.value = proj.filtro;
                    projectNameInput.value = proj.nombre_proyecto + " (copia)";
                    
                    fetch(proj.url_imagen)
                        .then(res => res.blob()) 
                        .then(blob => {
                            const reader = new FileReader();
                            reader.onload = (e) => {
                                const dataUrl = e.target.result;
                                originalImageDataUrl = dataUrl; 
                                loadUploadedImage(dataUrl, false);
                            };
                            reader.readAsDataURL(blob);

                            originalImageFile = new File([blob], proj.url_imagen.split('/').pop(), { type: blob.type });
                        });
                    
                    statusDiv.textContent = '¡Proyecto cargado con éxito!';
                } else {
                    statusDiv.textContent = `Error: ${data.message}`;
                }
            })
            .catch(err => {
                console.error('Error cargando proyecto:', err);
                statusDiv.textContent = 'Error de red al cargar el proyecto.';
            });
    }
}

// Inicializar la comprobación de sesión Y la carga de proyecto
document.addEventListener('DOMContentLoaded', () => {
    checkLoginStatus();
    checkAndLoadProject();
});


// --- Event Listeners ---

if (imageLoader) {
    imageLoader.addEventListener('change', (event) => {
        const file = event.target.files[0];
        if (!file) return;

        originalImageFile = file; // Guardar el archivo

        const reader = new FileReader();
        reader.onload = (e) => {
            originalImageDataUrl = e.target.result; 
            loadUploadedImage(originalImageDataUrl, true); // true = es una nueva subida
            if (imageToCrop) imageToCrop.src = originalImageDataUrl;
            if (cropBtn) cropBtn.style.display = 'block';
        };
        reader.readAsDataURL(file);
    });
}

// Modificada para saber si es una nueva subida o una carga
function loadUploadedImage(dataUrl, isNewUpload) {
    uploadedImage = new Image();
    uploadedImage.onload = () => {
        imageAspectRatio = uploadedImage.width / uploadedImage.height;
        if (imagePreview) imagePreview.src = dataUrl;
        if (imagePreview) imagePreview.style.display = 'block';
        if (previewPlaceholder) previewPlaceholder.style.display = 'none';
        if (previewContainer) previewContainer.style.display = 'flex';
        if (saveProjectContainer) saveProjectContainer.style.display = 'block'; 

        if (statusDiv) statusDiv.textContent = 'Imagen cargada. Define el tamaño del póster.';
        
        if (isNewUpload) { 
            if (aspectLock.checked) {
                posterWidthInput.value = 80;
                posterHeightInput.value = (80 / imageAspectRatio).toFixed(1);
            } else {
                posterWidthInput.value = 80;
                posterHeightInput.value = 60;
            }
            if (originalImageFile && projectNameInput) {
                projectNameInput.value = originalImageFile.name.split('.').slice(0, -1).join('.');
            }
        }
        updateGridPreview(); 
    };
    uploadedImage.onerror = () => {
        console.error("Error al cargar la imagen.");
        if (statusDiv) statusDiv.textContent = 'Error: El archivo subido parece estar corrupto o no es una imagen válida. Intenta con otro.';
        if (imagePreview) imagePreview.style.display = 'none';
        if (previewPlaceholder) previewPlaceholder.style.display = 'block';
        uploadedImage = null;
    };
    uploadedImage.src = dataUrl;
}

// Listeners de proporción y cuadrícula
if (posterWidthInput) {
    posterWidthInput.addEventListener('input', () => {
        if (aspectLock.checked && imageAspectRatio) {
            const widthCM = parseFloat(posterWidthInput.value);
            if (widthCM > 0) {
                posterHeightInput.value = (widthCM / imageAspectRatio).toFixed(1);
            }
        }
        updateGridPreview();
    });
}
if (posterHeightInput) {
    posterHeightInput.addEventListener('input', () => {
        if (aspectLock.checked && imageAspectRatio) {
            const heightCM = parseFloat(posterHeightInput.value);
            if (heightCM > 0) {
                posterWidthInput.value = (heightCM * imageAspectRatio).toFixed(1);
            }
        }
        updateGridPreview();
    });
}
if (paperSizeSelect) paperSizeSelect.addEventListener('input', updateGridPreview);
if (orientationSelect) orientationSelect.addEventListener('input', updateGridPreview);


// Lógica de Previsualización de Cuadrícula
function calculatePageLayout() {
    const posterWidthCM = parseFloat(posterWidthInput.value);
    const posterHeightCM = parseFloat(posterHeightInput.value);
    if (!posterWidthCM || !posterHeightCM || posterWidthCM <= 0 || posterHeightCM <= 0) {
        return { pagesWide: 0, pagesHigh: 0 };
    }
    const paperSize = paperSizeSelect.value;
    const orientation = orientationSelect.value;
    
    // Área imprimible (tamaño de papel menos márgenes de 1cm)
    let paperWidthMM = PAPER_SIZES_MM[paperSize].w - (PRINT_MARGIN_MM * 2);
    let paperHeightMM = PAPER_SIZES_MM[paperSize].h - (PRINT_MARGIN_MM * 2);
    
    if (orientation === 'landscape') {
        [paperWidthMM, paperHeightMM] = [paperHeightMM, paperWidthMM];
    }
    const posterWidthMM = posterWidthCM * 10;
    const posterHeightMM = posterHeightCM * 10;
    
    const pagesWide = Math.ceil(posterWidthMM / paperWidthMM);
    const pagesHigh = Math.ceil(posterHeightMM / paperHeightMM);
    return { pagesWide, pagesHigh };
}

function updateGridPreview() {
    if (!uploadedImage) return;
    const { pagesWide, pagesHigh } = calculatePageLayout();
    calculatedPagesWide = pagesWide;
    calculatedPagesHigh = pagesHigh;
    if (gridPreviewOverlay) gridPreviewOverlay.innerHTML = '';
    if (pagesWide <= 0 || pagesHigh <= 0 || pagesWide > 50 || pagesHigh > 50) {
        if (gridPreviewOverlay) gridPreviewOverlay.style.display = 'none';
        return;
    }
    if (gridPreviewOverlay) gridPreviewOverlay.style.display = 'flex';
    const totalCells = pagesWide * pagesHigh;
    const cellWidth = (100 / pagesWide).toFixed(4);
    const cellHeight = (100 / pagesHigh).toFixed(4);
    for (let i = 0; i < totalCells; i++) {
        const cell = document.createElement('div');
        cell.className = 'grid-cell';
        cell.style.width = `${cellWidth}%`;
        cell.style.height = `${cellHeight}%`;
        if (gridPreviewOverlay) gridPreviewOverlay.appendChild(cell);
    }
}


// Listeners de Recorte (Crop)
if (cropBtn) {
    cropBtn.addEventListener('click', () => {
        if (!originalImageDataUrl) return;
        imageToCrop.src = originalImageDataUrl;
        cropModal.style.display = 'flex';
        if (cropper) {
            cropper.destroy();
        }
        cropper = new Cropper(imageToCrop, {
            aspectRatio: NaN,
            viewMode: 1,
            background: false,
        });
    });
}
if (cancelCropBtn) {
    cancelCropBtn.addEventListener('click', () => {
        cropModal.style.display = 'none';
        if (cropper) {
            cropper.destroy();
        }
    });
}

// =================================================================
// --- ¡BLOQUE CORREGIDO PARA EL PROBLEMA DEL FONDO NEGRO! ---
// =================================================================
if (saveCropBtn) {
    saveCropBtn.addEventListener('click', () => {
        if (!cropper) return;
        const croppedCanvas = cropper.getCroppedCanvas({
            minWidth: 500,
            imageSmoothingEnabled: true,
            imageSmoothingQuality: 'high',
        });
        
        // CORRECCIÓN: Usar 'image/png' para mantener la transparencia
        const croppedDataUrl = croppedCanvas.toDataURL('image/png'); 
        
        croppedCanvas.toBlob((blob) => {
            // CORRECCIÓN: Guardar el archivo como .png y tipo 'image/png'
            originalImageFile = new File([blob], "recorte.png", { type: 'image/png' });
            loadUploadedImage(croppedDataUrl, true);
            originalImageDataUrl = croppedDataUrl;
        }, 'image/png'); // CORRECCIÓN: Usar 'image/png' para el blob
    
        cropModal.style.display = 'none';
        cropper.destroy();
    });
}
// =================================================================
// --- FIN DEL BLOQUE CORREGIDO ---
// =================================================================


// Listener del botón "Guardar Proyecto"
if (saveProjectBtn) {
    saveProjectBtn.addEventListener('click', () => {
        if (!originalImageFile) {
            alert('No hay una imagen válida para guardar.');
            return;
        }
        const nombre = projectNameInput.value;
        if (!nombre) {
            alert('Por favor, dale un nombre a tu proyecto.');
            return;
        }
        const ancho = parseFloat(posterWidthInput.value);
        const alto = parseFloat(posterHeightInput.value);
        if (!ancho || !alto || ancho <= 0 || alto <= 0) {
            alert('Por favor, define un tamaño válido (cm) para el póster.');
            return;
        }

        saveProjectBtn.disabled = true;
        saveProjectBtn.textContent = 'Guardando...';
        statusDiv.textContent = 'Guardando proyecto, por favor espera...';

        const formData = new FormData();
        formData.append('imagenProyecto', originalImageFile);
        formData.append('nombreProyecto', nombre);
        formData.append('ancho_cm', ancho);
        formData.append('alto_cm', alto);
        formData.append('filtro', filterSelect.value);
        formData.append('es_publico', makePublicCheckbox.checked ? '1' : '0');

        fetch('guardar_proyecto.php', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                statusDiv.textContent = '¡Proyecto guardado con éxito!';
                alert('¡Proyecto guardado con éxito!');
            } else {
                statusDiv.textContent = `Error al guardar: ${data.message}`;
                alert(`Error al guardar: ${data.message}`);
            }
            saveProjectBtn.disabled = false;
            saveProjectBtn.textContent = '💾 Guardar Proyecto';
        })
        .catch(error => {
            console.error('Error al guardar:', error);
            statusDiv.textContent = 'Error de red al guardar el proyecto.';
            alert('Error de red al guardar el proyecto.');
            saveProjectBtn.disabled = false;
            saveProjectBtn.textContent = '💾 Guardar Proyecto';
        });
    });
}


// Listener del botón "Generar PDF"
if (generateBtn) {
    generateBtn.addEventListener('click', () => {
        if (!uploadedImage) {
            alert('Por favor, sube una imagen primero.');
            return;
        }
        if (calculatedPagesWide <= 0 || calculatedPagesHigh <= 0) {
            alert('Por favor, introduce un ancho y alto válidos en centímetros.');
            return;
        }

        generateBtn.disabled = true;
        generateBtn.textContent = 'Procesando...';

        fetch('spend_credit.php', {
            method: 'POST',
            cache: 'no-store' 
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                updateCreditCounter(data.creditos_restantes);
                createPosterPDF(); 
                
                setTimeout(() => {
                    generateBtn.disabled = false;
                    generateBtn.textContent = '5. Generar PDF (Coste: 1 Crédito)';
                }, 3000); 

            } else {
                alert('¡No tienes créditos suficientes!');
                creditsModal.style.display = 'flex'; 
                generateBtn.disabled = false;
                generateBtn.textContent = '5. Generar PDF (Coste: 1 Crédito)';
            }
        })
        .catch(error => {
            console.error('Error al gastar el crédito:', error);
            alert('Error de conexión con el servidor de créditos. Inténtalo de nuevo.');
            generateBtn.disabled = false;
            generateBtn.textContent = '5. Generar PDF (Coste: 1 Crédito)';
        });
    });
}

// Listener del modal de créditos
if (closeCreditsModalBtn) {
    closeCreditsModalBtn.addEventListener('click', () => {
        creditsModal.style.display = 'none';
    });
}


// --- Función Principal de Generación de PDF (¡LÓGICA CORREGIDA!) ---
async function createPosterPDF() {
    statusDiv.textContent = 'Generando PDF... por favor espera.';
    
    const filterValue = filterSelect.value;
    if (filterValue === 'raster') {
        statusDiv.textContent = 'Aplicando efecto Raster... esto puede tardar varios segundos...';
    }
    
    setTimeout(async () => {
        try {
            const paperSize = paperSizeSelect.value;
            const orientation = orientationSelect.value;
            const addCutLines = addCutLinesCheckbox.checked;
            const addPageNumbers = addPageNumbersCheckbox.checked;
            const addSummaryPage = addSummaryPageCheckbox.checked;
            const addWatermark = addWatermarkCheckbox.checked;

            const pagesWide = calculatedPagesWide;
            const pagesHigh = calculatedPagesHigh;
            
            const posterWidthCM = parseFloat(posterWidthInput.value);
            const posterHeightCM = parseFloat(posterHeightInput.value);
            const posterWidthMM = posterWidthCM * 10;
            const posterHeightMM = posterHeightCM * 10;
            
            const doc = new jsPDF(orientation, 'mm', paperSize);
            const pdfPageWidth = doc.internal.pageSize.getWidth();
            const pdfPageHeight = doc.internal.pageSize.getHeight();
            
            // --- ¡LÓGICA DE IMPRESIÓN CORREGIDA! ---
            
            // 1. Área imprimible total de una hoja (Tamaño papel - 2cm de margen)
            const pagePrintableWidth = pdfPageWidth - (PRINT_MARGIN_MM * 2);
            const pagePrintableHeight = pdfPageHeight - (PRINT_MARGIN_MM * 2);
            
            // 2. Ancho y alto de la IMAGEN en cada hoja
            // Esto es el TAMAÑO REAL del póster dividido por el número de hojas
            const imageTileWidthMM = posterWidthMM / pagesWide;
            const imageTileHeightMM = posterHeightMM / pagesHigh;
            
            // 3. Márgenes para centrar la imagen en el área imprimible
            const marginX = PRINT_MARGIN_MM;
            const marginY = PRINT_MARGIN_MM;

            const imgWidth = uploadedImage.width;
            const imgHeight = uploadedImage.height;

            // Ancho y alto de CADA CORTE de la imagen original (en píxeles)
            const sliceWidthPx = imgWidth / pagesWide;
            const sliceHeightPx = imgHeight / pagesHigh;
            
            statusDiv.textContent = `Calculado: ${pagesWide}x${pagesHigh} hojas. Generando PDF...`;

            if (addSummaryPage) {
                await addPosterSummaryPage(doc, uploadedImage, pagesWide, pagesHigh, pdfPageWidth, pdfPageHeight, posterWidthCM, posterHeightCM, false);
            }

            for (let y = 0; y < pagesHigh; y++) {
                for (let x = 0; x < pagesWide; x++) {

                    if ((x > 0 || y > 0) || (addSummaryPage && x === 0 && y === 0)) {
                        doc.addPage(paperSize, orientation);
                    }

                    // --- CALCULAR TAMAÑO DE ESTA HOJA ESPECÍFICA ---
                    
                    // Coordenadas de corte (sx, sy) en píxeles
                    const sx = x * sliceWidthPx;
                    const sy = y * sliceHeightPx;
                    
                    // Ancho y alto del corte por defecto (en píxeles)
                    let sWidth = sliceWidthPx;
                    let sHeight = sliceHeightPx;
                    
                    // Ancho y alto de impresión por defecto (en mm)
                    let tilePrintWidth = imageTileWidthMM;
                    let tilePrintHeight = imageTileHeightMM;

                    // Ajustar si es la última columna
                    if (x === pagesWide - 1) {
                        const remainingWidthMM = posterWidthMM - (imageTileWidthMM * (pagesWide - 1));
                        tilePrintWidth = remainingWidthMM;
                        sWidth = (imgWidth / posterWidthMM) * tilePrintWidth;
                    }

                    // Ajustar si es la última fila
                    if (y === pagesHigh - 1) {
                        const remainingHeightMM = posterHeightMM - (imageTileHeightMM * (pagesHigh - 1));
                        tilePrintHeight = remainingHeightMM;
                        sHeight = (imgHeight / posterHeightMM) * tilePrintHeight;
                    }
                    // --- FIN DE CÁLCULO DE HOJA ---

                    const tileCanvas = document.createElement('canvas');
                    tileCanvas.width = sWidth;
                    tileCanvas.height = sHeight;
                    const ctx = tileCanvas.getContext('2d');

                    applyFilterAndEffects(ctx, filterValue, tileCanvas.width, tileCanvas.height);

                    // Dibujar solo el trozo de imagen en el canvas
                    ctx.drawImage(uploadedImage, sx, sy, sWidth, sHeight, 0, 0, sWidth, sHeight);
                    
                    if (filterValue === 'raster') {
                        applyRasterEffect(ctx, tileCanvas.width, tileCanvas.height, RASTER_DOT_SIZE);
                    }
                    
                    // CORRECCIÓN: Usar 'PNG' si la imagen original es PNG para mantener transparencia
                    const imageType = (originalImageFile.type === 'image/png') ? 'PNG' : 'JPEG';
                    const tileDataUrl = tileCanvas.toDataURL(originalImageFile.type || 'image/jpeg', 0.9);
                    
                    ctx.filter = 'none';
                    
                    // Dibujar la imagen DENTRO de los márgenes
                    doc.addImage(tileDataUrl, imageType, marginX, marginY, tilePrintWidth, tilePrintHeight);

                    // Dibujar las líneas de corte (si están activadas)
                    if (addCutLines) {
                        doc.setDrawColor(150, 150, 150); // Gris
                        doc.setLineDashPattern([2, 2], 0); // Punteado
                        // Dibuja un rectángulo alrededor de la imagen
                        doc.rect(marginX, marginY, tilePrintWidth, tilePrintHeight);
                    }
                    
                    if (addWatermark) {
                        doc.setFontSize(8);
                        doc.setTextColor(180, 180, 180);
                        doc.text('@rafooo90', pdfPageWidth - (PRINT_MARGIN_MM / 2), pdfPageHeight - (PRINT_MARGIN_MM / 2), { align: 'right' });
                    }

                    if (addPageNumbers) {
                        // Posición igual a la de PiñataPoster (Fila-Columna)
                        const pageText = `Fila ${y + 1}, Columna ${x + 1} (${String.fromCharCode(65 + y)}${x + 1})`;
                        doc.setFontSize(8);
                        doc.setTextColor(100, 100, 100);
                        doc.text(pageText, PRINT_MARGIN_MM, pdfPageHeight - (PRINT_MARGIN_MM / 2));
                    }
                }
            }

            doc.save('poster-gigante-rafon.pdf');
            statusDiv.textContent = `¡PDF de ${pagesWide}x${pagesHigh} hojas generado! Revisa tus descargas.`;

        } catch (error) {
            console.error('Error generando el PDF:', error);
            statusDiv.textContent = 'Error: No se pudo generar el PDF.';
            alert('Hubo un error al generar el PDF. Revisa la consola para más detalles.');
        }
    }, 50);
}

// --- Funciones de Ayuda ---

function applyFilterAndEffects(ctx, filterValue, width, height) {
    if (filterValue === 'grayscale') {
        ctx.filter = 'grayscale(100%)';
    } else if (filterValue === 'sepia') {
        ctx.filter = 'sepia(100%)';
    } else if (filterValue === 'raster') {
        ctx.fillStyle = 'white';
        ctx.fillRect(0, 0, width, height);
    } else {
        ctx.filter = 'none';
    }
}

function applyRasterEffect(ctx, width, height, dotSize) {
    const imageData = ctx.getImageData(0, 0, width, height);
    const data = imageData.data;
    ctx.fillStyle = 'white';
    ctx.fillRect(0, 0, width, height);
    ctx.fillStyle = 'black';
    for (let y = 0; y < height; y += dotSize) {
        for (let x = 0; x < width; x += dotSize) {
            let totalBrightness = 0;
            let pixelCount = 0;
            for (let blockY = 0; blockY < dotSize; blockY++) {
                for (let blockX = 0; blockX < dotSize; blockX++) {
                    const i = ((y + blockY) * width + (x + blockX)) * 4;
                    if (data[i]) {
                        const brightness = 0.299 * data[i] + 0.587 * data[i + 1] + 0.114 * data[i + 2];
                        totalBrightness += brightness;
                        pixelCount++;
                    }
                }
            }
            const avgBrightness = totalBrightness / pixelCount;
            const radius = (dotSize / 2) * (1 - (avgBrightness / 255));
            if (radius > 0) {
                ctx.beginPath();
                ctx.arc(x + dotSize / 2, y + dotSize / 2, radius, 0, 2 * Math.PI);
                ctx.fill();
            }
        }
    }
}

async function addPosterSummaryPage(doc, image, pagesWide, pagesHigh, pdfPageWidth, pdfPageHeight, posterWidthCM, posterHeightCM, hasOverlap) {
    const margin = 15;
    const contentWidth = pdfPageWidth - (2 * margin);
    const contentHeight = pdfPageHeight - (2 * margin);
    doc.setFontSize(24);
    doc.setTextColor(50, 50, 50);
    doc.text('Instrucciones para Armar tu Póster Gigante', margin, margin + 10);
    doc.setFontSize(12);
    doc.setTextColor(80, 80, 80);
    
    // Instrucciones actualizadas al método de corte
    let instructions = [
        `Tu póster de ${posterWidthCM}cm x ${posterHeightCM}cm se ha dividido en ${pagesHigh} filas por ${pagesWide} columnas.`,
        '1. ¡IMPORTANTE! Al imprimir, tu computadora debería seleccionar "Tamaño Real" o "100%" automáticamente.',
        '   (Si la imagen se ve cortada, SÍ puedes usar "Ajustar a página", los márgenes lo compensarán).', // ¡Instrucción más segura!
        '2. Identifica cada pieza usando los números (Fila-Columna) impresos en la esquina.',
        '3. CORTA con cuidado por las líneas punteadas que rodean cada imagen.',
        '4. Une las piezas por los bordes (como un rompecabezas) y pégalas con cinta por detrás.',
        '5. ¡Disfruta de tu póster gigante!'
    ];
    
    doc.text(instructions, margin, margin + 30, { maxWidth: contentWidth });
    const currentTextHeight = (instructions.length * 6); // Ajuste de altura
    const thumbMaxHeight = contentHeight - currentTextHeight - 30; // Más espacio
    let thumbWidth = contentWidth;
    let thumbHeight = thumbWidth / imageAspectRatio;
    if (thumbHeight > thumbMaxHeight) {
        thumbHeight = thumbMaxHeight;
        thumbWidth = thumbHeight * imageAspectRatio;
    }
    const thumbX = margin + (contentWidth - thumbWidth) / 2;
    const thumbY = pdfPageHeight - margin - thumbHeight;
    const imageToDraw = await convertImageToDataURL(image, thumbWidth, thumbHeight);
    doc.addImage(imageToDraw, 'JPEG', thumbX, thumbY, thumbWidth, thumbHeight);
    doc.setDrawColor(150, 150, 150);
    doc.setLineWidth(0.3);
    for (let i = 0; i <= pagesWide; i++) {
        const xPos = thumbX + (i * (thumbWidth / pagesWide));
        doc.line(xPos, thumbY, xPos, thumbY + thumbHeight);
    }
    for (let i = 0; i <= pagesHigh; i++) {
        const yPos = thumbY + (i * (thumbHeight / pagesHigh));
        doc.line(thumbX, yPos, thumbX + thumbWidth, yPos);
    }
    doc.setFontSize(9);
    doc.setTextColor(0, 0, 0);
    for (let y = 0; y < pagesHigh; y++) {
        for (let x = 0; x < pagesWide; x++) {
            const centerX = thumbX + (x * (thumbWidth / pagesWide)) + (thumbWidth / pagesWide / 2);
            const centerY = thumbY + (y * (thumbHeight / pagesHigh)) + (thumbHeight / pagesHigh / 2);
            doc.text(`${y + 1}-${x + 1}`, centerX, centerY, { align: 'center', baseline: 'middle' });
        }
    }
}

/**
 * Convierte una imagen a DataURL de un tamaño específico para la miniatura.
 */
function convertImageToDataURL(originalImage, targetWidth, targetHeight) {
    return new Promise((resolve) => {
        const canvas = document.createElement('canvas');
        canvas.width = targetWidth;
        canvas.height = targetHeight;
        const ctx = canvas.getContext('2d');
        ctx.drawImage(originalImage, 0, 0, targetWidth, targetHeight);
        resolve(canvas.toDataURL('image/jpeg', 0.8));
    });
}