// --- Lógica de Sesión ---
const creditCounter = document.getElementById('creditCounterDisplay');
const userSessionElement = document.getElementById('user-session');

function updateCreditCounter(amount) {
    if (creditCounter) {
        creditCounter.textContent = `Créditos: ${amount}`;
        creditCounter.style.display = 'block';
    }
}

function checkLoginStatus() {
    fetch('check_session.php', { cache: 'no-store' })
        .then(response => { if (!response.ok) throw new Error('Network error'); return response.json(); })
        .then(data => {
            if (data.loggedIn) {
                userSessionElement.innerHTML = `
                    <div class="user-info">
                        Bienvenido, <strong>${data.nombre}</strong> | <a href="index.html">Inicio</a> | <a href="galeria.html">Mi Galería</a> | <a href="logout.php">Cerrar Sesión</a>
                    </div>`;
                updateCreditCounter(data.creditos);
            } else {
                alert('Necesitas iniciar sesión.');
                window.location.href = 'index.html';
            }
        })
        .catch(e => console.error(e));
}
checkLoginStatus();


// --- Lógica de Banderines ---
document.addEventListener('DOMContentLoaded', () => {

    const canvas = document.getElementById('bannerCanvas');
    const ctx = canvas.getContext('2d');
    const bannerTextInput = document.getElementById('bannerText');
    const shapeSelector = document.getElementById('shapeSelector');
    const fontSelector = document.getElementById('fontSelector');
    const textColorPicker = document.getElementById('textColorPicker');
    const bgColorPicker = document.getElementById('bgColorPicker');
    const bgImageLoader = document.getElementById('bgImageLoader');
    const removeBgImageBtn = document.getElementById('removeBgImage');
    const bannerHeightInput = document.getElementById('bannerHeightInput'); // ¡Nuevo!
    const generatePdfBtn = document.getElementById('generatePdfBtn');
    const statusDiv = document.getElementById('status');

    let bgImage = null;

    // --- 1. Definir Forma ---
    function defineBannerShape(ctx, width, height, shape) {
        ctx.beginPath();
        if (shape === 'triangle') {
            ctx.moveTo(0, 0);
            ctx.lineTo(width, 0);
            ctx.lineTo(width / 2, height);
            ctx.closePath();
        } else if (shape === 'swallowtail') {
            ctx.moveTo(0, 0);
            ctx.lineTo(width, 0);
            ctx.lineTo(width, height);
            ctx.lineTo(width / 2, height - (height * 0.25));
            ctx.lineTo(0, height);
            ctx.closePath();
        } else if (shape === 'shield') {
            ctx.moveTo(0, 0);
            ctx.lineTo(width, 0);
            ctx.lineTo(width, height * 0.7);
            ctx.quadraticCurveTo(width / 2, height, 0, height * 0.7);
            ctx.closePath();
        } else {
            ctx.rect(0, 0, width, height);
        }
    }

    // --- 2. Función de Dibujo ---
    function drawBanner(letterToDraw) {
        const w = canvas.width;
        const h = canvas.height;
        const shape = shapeSelector.value;
        const font = fontSelector.value;
        const textColor = textColorPicker.value;
        const bgColor = bgColorPicker.value;

        ctx.clearRect(0, 0, w, h);

        ctx.save();
        
        // 1. Recortar forma
        defineBannerShape(ctx, w, h, shape);
        ctx.clip();

        // 2. ¡SOLUCIÓN AL FONDO NEGRO!
        // Siempre pintamos el fondo sólido primero.
        // Si el PNG es transparente, se verá este color.
        ctx.fillStyle = bgColor;
        ctx.fill(); 
        ctx.fillRect(0, 0, w, h); // Asegurar cobertura total

        // 3. Dibujar Imagen (si existe)
        if (bgImage) {
            const ratio = Math.max(w / bgImage.width, h / bgImage.height);
            const centerShift_x = (w - bgImage.width * ratio) / 2;
            const centerShift_y = (h - bgImage.height * ratio) / 2;
            ctx.drawImage(bgImage, 0, 0, bgImage.width, bgImage.height, centerShift_x, centerShift_y, bgImage.width * ratio, bgImage.height * ratio);
        }

        ctx.restore();

        // Borde guía
        ctx.save();
        defineBannerShape(ctx, w, h, shape);
        ctx.strokeStyle = '#cccccc';
        ctx.lineWidth = 2;
        ctx.stroke();
        ctx.restore();

        // 4. Letra
        if (letterToDraw) {
            ctx.fillStyle = textColor;
            ctx.font = `bold ${h * 0.5}px "${font}"`;
            ctx.textAlign = 'center';
            ctx.textBaseline = 'middle';
            ctx.shadowColor = "rgba(255, 255, 255, 0.8)";
            ctx.shadowBlur = 10;
            
            let yPos = h / 2;
            if (shape === 'triangle') yPos = h * 0.4;
            
            ctx.fillText(letterToDraw, w / 2, yPos);
        }
    }

    // --- Listeners ---
    [bannerTextInput, shapeSelector, fontSelector, textColorPicker, bgColorPicker].forEach(el => {
        el.addEventListener('input', () => {
            const text = bannerTextInput.value.trim() || "?";
            drawBanner(text[0].toUpperCase());
        });
    });

    bgImageLoader.addEventListener('change', (e) => {
        const file = e.target.files[0];
        if (!file) return;
        const reader = new FileReader();
        reader.onload = (event) => {
            bgImage = new Image();
            bgImage.onload = () => {
                removeBgImageBtn.style.display = 'inline-block';
                const text = bannerTextInput.value.trim() || "?";
                drawBanner(text[0].toUpperCase());
            };
            bgImage.src = event.target.result;
        };
        reader.readAsDataURL(file);
    });

    removeBgImageBtn.addEventListener('click', () => {
        bgImage = null;
        bgImageLoader.value = "";
        removeBgImageBtn.style.display = 'none';
        const text = bannerTextInput.value.trim() || "?";
        drawBanner(text[0].toUpperCase());
    });

    // --- GENERAR PDF (CON TAMAÑO PERSONALIZADO) ---
    generatePdfBtn.addEventListener('click', async () => {
        const text = bannerTextInput.value.trim();
        if (!text) {
            alert("Por favor escribe un texto para el letrero.");
            return;
        }

        generatePdfBtn.disabled = true;
        generatePdfBtn.textContent = "Generando...";
        statusDiv.textContent = "Calculando tamaños...";

        const { jsPDF } = window.jspdf;
        const doc = new jsPDF({ orientation: 'p', unit: 'mm', format: 'a4' });
        
        const letters = text.split('');
        const cleanLetters = letters.filter(char => char.trim() !== "");

        // --- LÓGICA DE TAMAÑO PERSONALIZADA ---
        // 1. Obtener altura deseada en CM y pasar a MM
        let desiredHeightCM = parseFloat(bannerHeightInput.value);
        if (!desiredHeightCM || desiredHeightCM < 5) desiredHeightCM = 20; // Default seguro
        const bannerHeightMM = desiredHeightCM * 10;

        // 2. Calcular el ancho basado en la proporción del Canvas original
        // (Así la imagen no se estira ni aplasta)
        const aspectRatio = canvas.width / canvas.height;
        const bannerWidthMM = bannerHeightMM * aspectRatio;

        // 3. Centrar en la hoja A4 (210mm ancho)
        const pdfPageWidth = 210; 
        const marginX = (pdfPageWidth - bannerWidthMM) / 2;
        const marginY = 20;

        for (let i = 0; i < cleanLetters.length; i++) {
            const char = cleanLetters[i].toUpperCase();
            
            drawBanner(char);
            
            // Usamos JPEG con calidad 1.0 (máxima). 
            // Como ya pintamos el fondo en el canvas, no habrá negro.
            const imgData = canvas.toDataURL('image/jpeg', 1.0);

            if (i > 0) doc.addPage();

            // Añadir imagen con las dimensiones calculadas
            doc.addImage(imgData, 'JPEG', marginX, marginY, bannerWidthMM, bannerHeightMM);
            
            // Pie de página
            doc.setFontSize(10);
            doc.setTextColor(150);
            doc.text(`Banderín ${i + 1}/${cleanLetters.length} - "${char}" - Alto: ${desiredHeightCM}cm`, 10, 280);
            doc.text(`espacioapp.site`, 200, 280, { align: 'right' });
        }

        doc.save(`Banderines_${text}.pdf`);
        
        statusDiv.textContent = "¡PDF Generado!";
        generatePdfBtn.disabled = false;
        generatePdfBtn.textContent = "📥 Generar PDF de Banderines";
    });

    drawBanner(bannerTextInput.value[0] || "A");
});