// --- Lógica de Sesión (Copiada de poster.js para consistencia) ---

const creditCounter = document.getElementById('creditCounterDisplay');
const userSessionElement = document.getElementById('user-session');

function updateCreditCounter(amount) {
    if (creditCounter) {
        creditCounter.textContent = `Créditos: ${amount}`;
        creditCounter.style.display = 'block';
    }
}

function checkLoginStatus() {
    fetch('check_session.php', { cache: 'no-store' })
        .then(response => {
            if (!response.ok) {
                throw new Error(`Error HTTP! estado: ${response.status}`);
            }
            return response.json();
        })
        .then(data => {
            if (data.loggedIn) {
                userSessionElement.innerHTML = `
                    <div class="user-info">
                        Bienvenido, <strong>${data.nombre}</strong>
                        <span>|</span>
                        <a href="index.html">Inicio</a>
                        <span>|</span>
                        <a href="galeria.html">Mi Galería</a>
                        <span>|</span>
                        <a href="comunidad.html">Galería Pública</a>
                        <span>|</span>
                        <a href="logout.php">Cerrar Sesión</a>
                    </div>
                `;
                updateCreditCounter(data.creditos);
            } else {
                alert('Necesitas iniciar sesión para usar esta herramienta.');
                window.location.href = 'index.html';
            }
        })
        .catch(error => {
            console.error('Error al chequear la sesión:', error);
            alert('Error de conexión o de sesión. Redirigiendo al inicio.');
            window.location.href = 'index.html';
        });
}

// --- Lógica de la Herramienta de Contornos ---

const imageLoader = document.getElementById('imageLoader');
const canvasInput = document.getElementById('canvasInput');
const canvasOutput = document.getElementById('canvasOutput');
const generateBtn = document.getElementById('generateBtn');
const downloadBtn = document.getElementById('downloadBtn');
const statusDiv = document.getElementById('status');

const threshold1Slider = document.getElementById('threshold1');
const threshold2Slider = document.getElementById('threshold2');
const threshold1Value = document.getElementById('threshold1Value');
const threshold2Value = document.getElementById('threshold2Value');

let img = new Image();
let isCvLoaded = false;
let isImageLoaded = false;

// ¡FUNCIÓN CORREGIDA!
// Esta función AHORA es llamada por el atributo 'onload' en la etiqueta <script> de contornos.html
function onCvReady() {
    // Ya no necesitamos 'if (cv)' porque 'onload' nos garantiza que existe.
    console.log('OpenCV.js está listo.');
    isCvLoaded = true;

    // Actualizar estado
    if (isImageLoaded) {
        // La imagen ya se había cargado, así que estamos listos
        statusDiv.textContent = '¡Todo listo! Ajusta los detalles y presiona "Generar".';
        generateBtn.disabled = false;
    } else {
        // La IA cargó primero, esperamos la imagen
        statusDiv.textContent = 'Biblioteca de IA cargada. Sube una imagen.';
    }
}

// Event Listeners para los Sliders
threshold1Slider.addEventListener('input', (e) => {
    threshold1Value.textContent = e.target.value;
});
threshold2Slider.addEventListener('input', (e) => {
    threshold2Value.textContent = e.target.value;
});

// 1. Cargar la imagen
imageLoader.addEventListener('change', (e) => {
    const file = e.target.files[0];
    if (!file) return;

    const reader = new FileReader();
    reader.onload = (event) => {
        img.src = event.target.result;
        img.onload = () => {
            // Mostrar imagen en el canvas de entrada
            const ctx = canvasInput.getContext('2d');
            canvasInput.width = img.width;
            canvasInput.height = img.height;
            ctx.drawImage(img, 0, 0);

            isImageLoaded = true;

            // Habilitar el botón solo si OpenCV también está cargada
            if (isCvLoaded) {
                statusDiv.textContent = '¡Todo listo! Ajusta los detalles y presiona "Generar".';
                generateBtn.disabled = false;
            } else {
                // Este es el estado que veías
                statusDiv.textContent = 'Imagen cargada. Esperando la biblioteca de IA... (puede tardar un momento)';
            }

            // Limpiar el canvas de salida
            const ctxOut = canvasOutput.getContext('2d');
            ctxOut.clearRect(0, 0, canvasOutput.width, canvasOutput.height);
            downloadBtn.style.display = 'none';
        };
    };
    reader.readAsDataURL(file);
});

// 2. Generar el Contorno
generateBtn.addEventListener('click', () => {
    if (!isImageLoaded || !isCvLoaded) {
        alert('Por favor, carga una imagen y espera a que la biblioteca esté lista.');
        return;
    }

    statusDiv.textContent = 'Procesando imagen...';
    generateBtn.disabled = true;

    // Usamos un setTimeout para que el navegador actualice el 'statusDiv'
    setTimeout(() => {
        try {
            // Cargar la imagen del canvas a OpenCV
            let src = cv.imread(canvasInput);
            let gray = new cv.Mat();
            let edges = new cv.Mat();

            // 1. Convertir a escala de grises
            cv.cvtColor(src, gray, cv.COLOR_RGBA2GRAY, 0);

            // 2. Aplicar el detector de bordes Canny
            let t1 = parseInt(threshold1Slider.value, 10);
            let t2 = parseInt(threshold2Slider.value, 10);
            cv.Canny(gray, edges, t1, t2);

            // 3. Invertir los colores (Canny da líneas blancas sobre fondo negro)
            // ¡Queremos líneas negras sobre fondo blanco para el molde!
            cv.bitwise_not(edges, edges);

            // Mostrar el resultado en el canvas de salida
            cv.imshow(canvasOutput, edges);

            // Limpiar la memoria (¡MUY IMPORTANTE en OpenCV.js!)
            src.delete();
            gray.delete();
            edges.delete();

            statusDiv.textContent = '¡Contorno generado con éxito!';

            // Preparar el botón de descarga
            downloadBtn.href = canvasOutput.toDataURL('image/png');
            downloadBtn.download = 'contorno_espacioapp.png';
            downloadBtn.style.display = 'inline-block';

        } catch (err) {
            console.error("Error en OpenCV:", err);
            statusDiv.textContent = 'Error al procesar la imagen. Revisa la consola.';
        } finally {
            generateBtn.disabled = false;
        }
    }, 50); // 50ms de espera
});

// --- INICIALIZACIÓN ---
checkLoginStatus(); // Revisa la sesión
// Ya no llamamos a onCvReady() desde aquí, el HTML se encarga.