// Asegurarnos de que jsPDF esté disponible globalmente
const { jsPDF } = window.jspdf;

// --- Constantes de Tamaño de Papel (en milímetros) ---
const PAPER_SIZES_MM = {
    a4: { w: 210, h: 297 },
    letter: { w: 215.9, h: 279.4 }
};
const RASTER_DOT_SIZE = 8; 
const PRINT_MARGIN_MM = 10; 

// --- DOM Elements ---
const pageSubtitle = document.getElementById('page-subtitle');
const imageLoader = document.getElementById('imageLoader');
const imagePreview = document.getElementById('imagePreview');
const previewContainer = document.getElementById('preview-container');
const previewPlaceholder = document.getElementById('previewPlaceholder');
const gridPreviewOverlay = document.getElementById('gridPreviewOverlay'); 

const filterSelect = document.getElementById('filterSelect'); 
const posterWidthInput = document.getElementById('posterWidth');
const posterHeightInput = document.getElementById('posterHeight');
const aspectLock = document.getElementById('aspectLock');

const paperSizeSelect = document.getElementById('paperSize');
const orientationSelect = document.getElementById('orientation');

const addCutLinesCheckbox = document.getElementById('addCutLines');
const addPageNumbersCheckbox = document.getElementById('addPageNumbers');
const addSummaryPageCheckbox = document.getElementById('addSummaryPage');
const addWatermarkCheckbox = document.getElementById('addWatermark');

const generateBtn = document.getElementById('generateBtn');
const statusDiv = document.getElementById('status');
const creditCounter = document.getElementById('creditCounterDisplay'); 
const userSessionElement = document.getElementById('user-session'); 

const cropBtn = document.getElementById('cropBtn');
const cropModal = document.getElementById('cropModal');
const imageToCrop = document.getElementById('imageToCrop');
const saveCropBtn = document.getElementById('saveCropBtn');
const cancelCropBtn = document.getElementById('cancelCropBtn');
let cropper = null;

const creditsModal = document.getElementById('creditsModal');
const closeCreditsModalBtn = document.getElementById('closeCreditsModalBtn');

const saveProjectContainer = document.getElementById('save-project-box');
const projectNameInput = document.getElementById('projectName');
const makePublicCheckbox = document.getElementById('makePublicCheckbox');
const saveProjectBtn = document.getElementById('saveProjectBtn');

// --- Variables Globales ---
let uploadedImage = null; 
// NOTA IMPORTANTE: originalImageDataUrl MANTIENE LA ALTA RESOLUCIÓN
let originalImageDataUrl = null; 
let originalImageFile = null; 
let imageAspectRatio = null;
let calculatedPagesWide = 0;
let calculatedPagesHigh = 0;

// --- FUNCIÓN AUTO-TRIM ---
function autoTrimImage(dataUrl) {
    return new Promise((resolve) => {
        const img = new Image();
        img.crossOrigin = "Anonymous"; 
        img.onload = () => {
            const canvas = document.createElement('canvas');
            const ctx = canvas.getContext('2d');
            canvas.width = img.width;
            canvas.height = img.height;
            ctx.drawImage(img, 0, 0);
            const w = canvas.width; const h = canvas.height;
            const pix = ctx.getImageData(0, 0, w, h).data;
            let top = null, bottom = null, left = null, right = null;
            for (let y = 0; y < h; y++) {
                for (let x = 0; x < w; x++) {
                    const i = (y * w + x) * 4;
                    const isTrans = (pix[i+3] === 0);
                    const isWhite = (pix[i]>240 && pix[i+1]>240 && pix[i+2]>240 && pix[i+3]>0);
                    if (!isTrans && !isWhite) {
                        if (top === null) top = y; bottom = y;
                        if (left === null || x < left) left = x;
                        if (right === null || x > right) right = x;
                    }
                }
            }
            if (top === null) { resolve(dataUrl); return; }
            const p = 2;
            top = Math.max(0, top - p); bottom = Math.min(h, bottom + p);
            left = Math.max(0, left - p); right = Math.min(w, right + p);
            const tW = right - left; const tH = bottom - top;
            if (tW <= 0 || tH <= 0) { resolve(dataUrl); return; }
            const tC = document.createElement('canvas');
            tC.width = tW; tC.height = tH;
            tC.getContext('2d').drawImage(img, left, top, tW, tH, 0, 0, tW, tH);
            resolve(tC.toDataURL('image/png'));
        };
        img.onerror = () => resolve(dataUrl);
        img.src = dataUrl;
    });
}

// --- SESIÓN ---
function updateCreditCounter(amount) {
    if (creditCounter) { creditCounter.textContent = `Créditos: ${amount}`; creditCounter.style.display = 'block'; }
}
function checkLoginStatus() {
    fetch('check_session.php', { cache: 'no-store' }) 
        .then(r => r.ok ? r.json() : Promise.reject(r))
        .then(data => {
            if (data.loggedIn) {
                userSessionElement.innerHTML = `<div class="user-info">Hola, <strong>${data.nombre}</strong> | <a href="index.html">Inicio</a> | <a href="galeria.html">Galería</a> | <a href="logout.php">Salir</a></div>`;
                updateCreditCounter(data.creditos);
            }
        }).catch(e => console.error(e));
}

// --- CARGA DE PROYECTO ---
function checkAndLoadProject() {
    const urlParams = new URLSearchParams(window.location.search);

    // 1. Carga desde IA
    if (urlParams.get('action') === 'load_temp') {
        const tempImg = sessionStorage.getItem('temp_poster_image');
        if (tempImg) {
            statusDiv.textContent = "Procesando imagen de IA...";
            fetch(tempImg).then(r=>r.blob()).then(blob=>{
                const reader = new FileReader();
                reader.onload = async (e) => {
                    const trimmed = await autoTrimImage(e.target.result);
                    fetch(trimmed).then(r=>r.blob()).then(b=>{
                        originalImageFile = new File([b], "ia_image.png", {type:"image/png"});
                        originalImageDataUrl = trimmed;
                        loadUploadedImage(trimmed, true);
                    });
                };
                reader.readAsDataURL(blob);
            });
            sessionStorage.removeItem('temp_poster_image');
            return;
        }
    }

    // 2. Carga desde Galería
    const projectId = urlParams.get('cargar_id');
    if (projectId) {
        statusDiv.textContent = 'Buscando proyecto...';
        imageLoader.disabled = true;

        fetch(`cargar_proyecto.php?id=${projectId}`, { cache: 'no-store' })
            .then(r => r.json())
            .then(data => {
                if (data.success && data.proyecto) {
                    const proj = data.proyecto;
                    statusDiv.textContent = 'Cargando datos...';
                    
                    if(projectNameInput) projectNameInput.value = proj.nombre_proyecto;
                    posterWidthInput.value = parseFloat(proj.ancho_cm);
                    posterHeightInput.value = parseFloat(proj.alto_cm);
                    filterSelect.value = proj.filtro || 'none';

                    originalImageDataUrl = proj.url_imagen;
                    loadUploadedImage(proj.url_imagen, false);
                    statusDiv.textContent = '¡Proyecto cargado!';

                    fetch(proj.url_imagen).then(res => res.blob()).then(blob => {
                        originalImageFile = new File([blob], "proyecto.png", { type: blob.type });
                    });
                } else {
                    statusDiv.textContent = `Error: ${data.message}`;
                    imageLoader.disabled = false;
                }
            })
            .catch(err => {
                console.error(err);
                statusDiv.textContent = 'Error de conexión.';
                imageLoader.disabled = false;
            });
    }
}

document.addEventListener('DOMContentLoaded', () => {
    checkLoginStatus();
    checkAndLoadProject();
});

// --- LISTENERS ---

if (imageLoader) {
    imageLoader.addEventListener('change', async (ev) => {
        const file = ev.target.files[0];
        if(!file) return;
        statusDiv.textContent = "Analizando imagen...";
        const reader = new FileReader();
        reader.onload = async (e) => {
            let res = e.target.result;
            try { res = await autoTrimImage(res); } catch(err){}
            fetch(res).then(r=>r.blob()).then(b=>{
                originalImageFile = new File([b], file.name, {type:'image/png'});
                originalImageDataUrl = res;
                loadUploadedImage(res, true);
                if(imageToCrop) imageToCrop.src = res;
                if(cropBtn) cropBtn.style.display='block';
                statusDiv.textContent = "Imagen lista. Define tamaño.";
            });
        };
        reader.readAsDataURL(file);
    });
}

function loadUploadedImage(src, isNew) {
    uploadedImage = new Image();
    uploadedImage.crossOrigin = "Anonymous"; 
    uploadedImage.onload = () => {
        imageAspectRatio = uploadedImage.width / uploadedImage.height;
        if(imagePreview) { imagePreview.src = src; imagePreview.style.display = 'block'; }
        if(previewPlaceholder) previewPlaceholder.style.display = 'none';
        if(previewContainer) previewContainer.style.display = 'flex';
        if(saveProjectContainer) saveProjectContainer.style.display = 'block';

        if(isNew) {
            if(aspectLock.checked) {
                posterWidthInput.value = 60;
                posterHeightInput.value = (60 / imageAspectRatio).toFixed(1);
            } else {
                posterWidthInput.value = 60; posterHeightInput.value = 60;
            }
        }
        updateGridPreview(); 
    };
    uploadedImage.src = src;
}

if(posterWidthInput) posterWidthInput.addEventListener('input', ()=>{
    if(aspectLock.checked && imageAspectRatio) {
        const v = parseFloat(posterWidthInput.value);
        if(v>0) posterHeightInput.value = (v/imageAspectRatio).toFixed(1);
    }
    updateGridPreview();
});
if(posterHeightInput) posterHeightInput.addEventListener('input', ()=>{
    if(aspectLock.checked && imageAspectRatio) {
        const v = parseFloat(posterHeightInput.value);
        if(v>0) posterWidthInput.value = (v*imageAspectRatio).toFixed(1);
    }
    updateGridPreview();
});
if(paperSizeSelect) paperSizeSelect.addEventListener('input', updateGridPreview);
if(orientationSelect) orientationSelect.addEventListener('input', updateGridPreview);

function calculateLayout() {
    const w = parseFloat(posterWidthInput.value);
    const h = parseFloat(posterHeightInput.value);
    if(!w || !h) return {w:0, h:0};
    const paper = paperSizeSelect.value;
    let pw = PAPER_SIZES_MM[paper].w - (PRINT_MARGIN_MM*2);
    let ph = PAPER_SIZES_MM[paper].h - (PRINT_MARGIN_MM*2);
    if(orientationSelect.value==='landscape') [pw, ph] = [ph, pw];
    return { cols: Math.ceil((w*10)/pw), rows: Math.ceil((h*10)/ph) };
}

function updateGridPreview() {
    if(!uploadedImage) return;
    const {cols, rows} = calculateLayout();
    calculatedPagesWide = cols; calculatedPagesHigh = rows;
    if(gridPreviewOverlay) {
        gridPreviewOverlay.innerHTML = '';
        if(cols>0 && cols<50) {
            gridPreviewOverlay.style.display='flex';
            const cw = (100/cols).toFixed(4); const ch = (100/rows).toFixed(4);
            for(let i=0; i<cols*rows; i++){
                const d = document.createElement('div');
                d.className='grid-cell'; d.style.width=`${cw}%`; d.style.height=`${ch}%`;
                gridPreviewOverlay.appendChild(d);
            }
        } else { gridPreviewOverlay.style.display='none'; }
    }
}

if(cropBtn) cropBtn.addEventListener('click', ()=>{
    if(!originalImageDataUrl) return;
    imageToCrop.src = originalImageDataUrl;
    cropModal.style.display='flex';
    if(cropper) cropper.destroy();
    cropper = new Cropper(imageToCrop, {viewMode:1, background:false});
});
if(cancelCropBtn) cancelCropBtn.addEventListener('click', ()=>{
    cropModal.style.display='none'; if(cropper) cropper.destroy();
});
if(saveCropBtn) saveCropBtn.addEventListener('click', ()=>{
    if(!cropper) return;
    const cvs = cropper.getCroppedCanvas({minWidth:500});
    const url = cvs.toDataURL('image/png');
    cvs.toBlob(blob=>{
        originalImageFile = new File([blob], "recorte.png", {type:'image/png'});
        loadUploadedImage(url, true);
        originalImageDataUrl = url;
    }, 'image/png');
    cropModal.style.display='none'; cropper.destroy();
});

if(saveProjectBtn) saveProjectBtn.addEventListener('click', ()=>{
    if(!originalImageFile || !projectNameInput.value) { alert('Falta imagen o nombre.'); return; }
    saveProjectBtn.disabled=true; saveProjectBtn.textContent='Guardando...';
    const fd = new FormData();
    fd.append('imagenProyecto', originalImageFile);
    fd.append('nombreProyecto', projectNameInput.value);
    fd.append('ancho_cm', posterWidthInput.value);
    fd.append('alto_cm', posterHeightInput.value);
    fd.append('filtro', filterSelect.value);
    fd.append('es_publico', makePublicCheckbox.checked?'1':'0');
    
    fetch('guardar_proyecto.php', {method:'POST', body:fd})
        .then(r=>r.json()).then(d=>{
            alert(d.message);
            saveProjectBtn.disabled=false; saveProjectBtn.textContent='💾 Guardar Proyecto';
        }).catch(()=>{
            alert('Error de red.'); saveProjectBtn.disabled=false;
        });
});

if(generateBtn) generateBtn.addEventListener('click', ()=>{
    if(!uploadedImage) { alert('Falta imagen.'); return; }
    generateBtn.disabled=true; generateBtn.textContent='Procesando...';
    fetch('spend_credit.php', {method:'POST', cache:'no-store'})
        .then(r=>r.json()).then(d=>{
            if(d.success) {
                updateCreditCounter(d.creditos_restantes);
                createPosterPDF();
                setTimeout(()=>{generateBtn.disabled=false; generateBtn.textContent='5. Generar PDF (Coste: 1 Crédito)';}, 3000);
            } else {
                alert('Sin créditos.'); creditsModal.style.display='flex'; generateBtn.disabled=false;
            }
        }).catch(()=>{ alert('Error conexión.'); generateBtn.disabled=false; });
});
if(closeCreditsModalBtn) closeCreditsModalBtn.addEventListener('click', ()=>creditsModal.style.display='none');

// --- GENERACIÓN PDF FINAL ---
async function createPosterPDF() {
    statusDiv.textContent='Generando PDF...';
    setTimeout(async ()=>{
        try {
            const paper = paperSizeSelect.value;
            const orient = orientationSelect.value;
            const wCM = parseFloat(posterWidthInput.value);
            const hCM = parseFloat(posterHeightInput.value);
            const {cols, rows} = calculateLayout();
            const wMM = wCM*10; const hMM = hCM*10;
            
            const doc = new jsPDF(orient, 'mm', paper);
            const pw = doc.internal.pageSize.getWidth();
            const ph = doc.internal.pageSize.getHeight();
            const printMargin = PRINT_MARGIN_MM;
            
            const tileW = wMM / cols; const tileH = hMM / rows;
            const sliceW = uploadedImage.width / cols; const sliceH = uploadedImage.height / rows;

            // 1. Página de Instrucciones
            if(addSummaryPageCheckbox.checked) await addPosterSummaryPage(doc, uploadedImage, cols, rows, pw, ph, wCM, hCM);

            for(let y=0; y<rows; y++) {
                for(let x=0; x<cols; x++) {
                    if(x>0||y>0||addSummaryPageCheckbox.checked) doc.addPage(paper, orient);
                    
                    const cvs = document.createElement('canvas');
                    cvs.width = sliceW; cvs.height = sliceH;
                    const ctx = cvs.getContext('2d');
                    
                    const filter = filterSelect.value;
                    if(filter==='grayscale') ctx.filter='grayscale(100%)';
                    else if(filter==='sepia') ctx.filter='sepia(100%)';
                    else if(filter==='raster') { ctx.fillStyle='white'; ctx.fillRect(0,0,cvs.width,cvs.height); }
                    
                    ctx.drawImage(uploadedImage, x*sliceW, y*sliceH, sliceW, sliceH, 0, 0, cvs.width, cvs.height);
                    if(filter==='raster') applyRasterEffect(ctx, cvs.width, cvs.height, RASTER_DOT_SIZE);

                    const imgData = cvs.toDataURL('image/jpeg', 0.95);
                    doc.addImage(imgData, 'JPEG', printMargin, printMargin, tileW, tileH);
                    
                    if(addCutLinesCheckbox.checked) {
                        doc.setDrawColor(150); doc.setLineDash([2,2]);
                        doc.rect(printMargin, printMargin, tileW, tileH);
                    }
                    if(addPageNumbersCheckbox.checked) {
                        doc.setFontSize(8); doc.setTextColor(100);
                        doc.text(`Fila ${y+1}-${x+1}`, printMargin, ph-5);
                    }
                    // 2. Marca de Agua
                    if(addWatermarkCheckbox.checked) {
                        doc.setFontSize(8); doc.setTextColor(180, 180, 180);
                        doc.text('TikTok: @rafooo90', pw - (printMargin/2), ph - (printMargin/2), { align: 'right' });
                    }
                }
            }
            doc.save('Poster_Gigante.pdf');
            statusDiv.textContent='¡Listo!';
        } catch(e) { console.error(e); statusDiv.textContent='Error en PDF.'; }
    }, 100);
}

function applyRasterEffect(ctx, w, h, size) {
    const d = ctx.getImageData(0,0,w,h).data;
    ctx.fillStyle='white'; ctx.fillRect(0,0,w,h); ctx.fillStyle='black';
    for(let y=0; y<h; y+=size) {
        for(let x=0; x<w; x+=size) {
            const i=(y*w+x)*4; const b=(d[i]*0.3+d[i+1]*0.59+d[i+2]*0.11);
            const r=(size/2)*(1-b/255);
            if(r>0){ ctx.beginPath(); ctx.arc(x+size/2, y+size/2, r, 0, Math.PI*2); ctx.fill(); }
        }
    }
}

// --- CORRECCIÓN DE CALIDAD DE MINIATURA ---
async function addPosterSummaryPage(doc, img, cols, rows, pw, ph, wCM, hCM) {
    doc.setFontSize(18); doc.setTextColor(50,50,50);
    doc.text("Instrucciones para Armar tu Póster Gigante", 15, 20);
    doc.setFontSize(11); doc.setTextColor(80,80,80);
    
    const instructions = [
        `Tu póster de ${wCM}cm x ${hCM}cm se ha dividido en ${rows} filas por ${cols} columnas.`,
        '1. ¡IMPORTANTE! Al imprimir, usa "Tamaño Real" o "100%".',
        '   (Si se corta, puedes usar "Ajustar a página", los márgenes lo compensarán).',
        '2. Identifica cada pieza usando los números (Fila-Columna).',
        '3. CORTA por las líneas punteadas.',
        '4. Une las piezas y pega con cinta por detrás.',
        '5. ¡Disfruta de tu póster gigante! - TikTok: @rafooo90'
    ];
    doc.text(instructions, 15, 30);
    
    // Cálculo de espacio disponible en la hoja
    const maxW = pw-40; const maxH = ph-100;
    const ratio = Math.min(maxW/img.width, maxH/img.height);
    // Dimensiones visuales en el PDF (mm)
    const tw = img.width*ratio; const th = img.height*ratio;
    
    // --- SOLUCIÓN DE CALIDAD ---
    // Usamos directamente la imagen original de alta resolución (originalImageDataUrl)
    // y dejamos que el PDF se encargue de escalarla visualmente, sin perder calidad.
    const format = (originalImageFile && originalImageFile.type === 'image/png') ? 'PNG' : 'JPEG';
    doc.addImage(originalImageDataUrl, format, 20, 70, tw, th);
    
    // Dibujar la cuadrícula sobre la miniatura
    doc.setDrawColor(50); doc.setLineWidth(0.1);
    for(let i=0; i<=cols; i++) doc.line(20+(i*tw/cols), 70, 20+(i*tw/cols), 70+th);
    for(let i=0; i<=rows; i++) doc.line(20, 70+(i*th/rows), 20+tw, 70+(i*th/rows));
}